<?php
include_once __DIR__ . '/auth_check.php';
include_once __DIR__ . '/config/db.php';

mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

$adminId = $_SESSION['admin_id'];
$role    = $_SESSION['admin_role'];

$message = '';
$error   = '';

if (!isset($_GET['customer_id'])) {
    die("Invalid request");
}

$customer_id = (int)$_GET['customer_id'];
$billMonth   = (int)date('n');
$billYear    = (int)date('Y');

/* ================= FETCH CUSTOMER + CURRENT BILL ================= */
if ($role === 'SUPER') {

    $sql = "
        SELECT 
            b.id AS bill_id,
            b.amount,
            b.paid_amount,
            b.status,
            c.Name AS customer_name,
            IFNULL(c.carry_balance,0) AS carry_balance
        FROM bills b
        JOIN customers c ON b.customer_id = c.id
        WHERE b.customer_id = ?
          AND b.bill_month = ?
          AND b.bill_year  = ?
    ";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("iii", $customer_id, $billMonth, $billYear);

} else {

    $sql = "
        SELECT 
            b.id AS bill_id,
            b.amount,
            b.paid_amount,
            b.status,
            c.Name AS customer_name,
            IFNULL(c.carry_balance,0) AS carry_balance
        FROM bills b
        JOIN customers c ON b.customer_id = c.id
        WHERE b.customer_id = ?
          AND b.bill_month = ?
          AND b.bill_year  = ?
          AND c.admin_id   = ?
    ";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("iiii", $customer_id, $billMonth, $billYear, $adminId);
}

$stmt->execute();
$data = $stmt->get_result()->fetch_assoc();
$stmt->close();

if (!$data) {
    die("Bill not generated for this month");
}

/* ================= HANDLE PAYMENT ================= */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    $newPayment = (float)$_POST['paid_amount'];

    if ($newPayment <= 0) {
        $error = "Invalid payment amount";
    } else {

        $billAmount = (float)$data['amount'];
        $totalPaid  = (float)$data['paid_amount'] + $newPayment;

        if ($totalPaid >= $billAmount) {
            $status = 'PAID';
        } elseif ($totalPaid > 0) {
            $status = 'PARTIAL';
        } else {
            $status = 'UNPAID';
        }

        $carryUpdate = ($totalPaid > $billAmount)
            ? ($totalPaid - $billAmount)
            : -($billAmount - $totalPaid);

        try {
            $conn->begin_transaction();

            /* 1️⃣ INSERT PAYMENT (ADMIN SESSION) */
            $payStmt = $conn->prepare("
                INSERT INTO payments
                (
                    bill_id,
                    amount,
                    payment_date,
                    mode,
                    entered_by,
                    admin_id,
                    officer_id
                )
                VALUES
                (?, ?, NOW(), 'CASH', ?, ?, NULL)
            ");
            $payStmt->bind_param(
                "idii",
                $data['bill_id'],
                $newPayment,
                $adminId,   // entered_by
                $adminId    // admin_id
            );
            $payStmt->execute();
            $payStmt->close();

            /* 2️⃣ UPDATE BILL */
            $upBill = $conn->prepare("
                UPDATE bills
                SET paid_amount = ?, status = ?
                WHERE id = ?
            ");
            $upBill->bind_param("dsi", $totalPaid, $status, $data['bill_id']);
            $upBill->execute();
            $upBill->close();

            /* 3️⃣ UPDATE CUSTOMER */
            $upCust = $conn->prepare("
                UPDATE customers
                SET carry_balance = ?
                WHERE id = ?
            ");
            $upCust->bind_param("di", $carryUpdate, $customer_id);
            $upCust->execute();
            $upCust->close();

            /* 4️⃣ INSERT PAYMENT LOG */
            $logStmt = $conn->prepare("
                INSERT INTO payment_logs
                (
                    bill_id,
                    customer_id,
                    amount,
                    action,
                    entered_by,
                    role,
                    mode,
                    admin_id
                )
                VALUES
                (?, ?, ?, 'PAYMENT', ?, ?, 'CASH', ?)
            ");
            $logStmt->bind_param(
                "iidisi",
                $data['bill_id'],
                $customer_id,
                $newPayment,
                $adminId,
                $role,
                $adminId
            );
            $logStmt->execute();
            $logStmt->close();

            $conn->commit();

            $message = "Payment recorded successfully";

            $data['paid_amount']   = $totalPaid;
            $data['status']        = $status;
            $data['carry_balance'] = $carryUpdate;

        } catch (Exception $e) {
            $conn->rollback();
            $error = $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <title>Make Payment</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body>

<div class="container mt-4">
    <h3>Make Payment</h3>

    <p>
        <b>Customer:</b> <?= htmlspecialchars($data['customer_name']) ?><br>
        <b>Bill Amount:</b> ₹<?= number_format($data['amount'],2) ?><br>
        <b>Paid So Far:</b> ₹<?= number_format($data['paid_amount'],2) ?><br>
        <b>Status:</b> <?= $data['status'] ?>
    </p>

    <?php if ($message): ?>
        <div class="alert alert-success"><?= $message ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="alert alert-danger"><?= $error ?></div>
    <?php endif; ?>

    <form method="post">
        <label>Enter Payment Amount</label>
        <input type="number" step="0.01" name="paid_amount"
               class="form-control mb-3" required>

        <button class="btn btn-success">Submit Payment</button>
        <a href="customers.php" class="btn btn-secondary">Back</a>
    </form>
</div>

</body>
</html>
