<?php
header("Content-Type: application/json");

include_once "../admin/config/db.php";
include_once "jwt.php";

/*
|--------------------------------------------------------------------------
| JWT AUTH (SAME AS CUSTOMER API)
|--------------------------------------------------------------------------
*/
$headers = array_change_key_case(getallheaders(), CASE_LOWER);
$auth = $headers['authorization'] ?? '';

if (!$auth || !preg_match('/Bearer\s(\S+)/', $auth, $matches)) {
    http_response_code(401);
    echo json_encode([
        "success" => false,
        "message" => "Unauthorized"
    ]);
    exit;
}

$token = $matches[1];
$user  = verify_jwt($token);

if (!$user || empty($user['role'])) {
    http_response_code(401);
    echo json_encode([
        "success" => false,
        "message" => "Invalid or expired token"
    ]);
    exit;
}

/*
|--------------------------------------------------------------------------
| ROLE CONTEXT
|--------------------------------------------------------------------------
*/
$role    = $user['role'];
$adminId = $user['admin_id'] ?? null;

$billMonth = (int)date('n');
$billYear  = (int)date('Y');

/*
|--------------------------------------------------------------------------
| TOTAL CUSTOMERS
|--------------------------------------------------------------------------
*/
if ($role === 'SUPER') {
    $totalCustomers = $conn->query(
        "SELECT COUNT(*) total FROM customers"
    )->fetch_assoc()['total'];
} else {
    $stmt = $conn->prepare(
        "SELECT COUNT(*) total FROM customers WHERE admin_id=?"
    );
    $stmt->bind_param("i", $adminId);
    $stmt->execute();
    $totalCustomers = $stmt->get_result()->fetch_assoc()['total'];
}

/*
|--------------------------------------------------------------------------
| ACTIVE CUSTOMERS
|--------------------------------------------------------------------------
*/
if ($role === 'SUPER') {
    $activeCustomers = $conn->query(
        "SELECT COUNT(*) total FROM customers WHERE Status='Active'"
    )->fetch_assoc()['total'];
} else {
    $stmt = $conn->prepare(
        "SELECT COUNT(*) total FROM customers WHERE Status='Active' AND admin_id=?"
    );
    $stmt->bind_param("i", $adminId);
    $stmt->execute();
    $activeCustomers = $stmt->get_result()->fetch_assoc()['total'];
}

/*
|--------------------------------------------------------------------------
| INACTIVE CUSTOMERS
|--------------------------------------------------------------------------
*/
if ($role === 'SUPER') {
    $inactiveCustomers = $conn->query(
        "SELECT COUNT(*) total FROM customers WHERE Status!='Active'"
    )->fetch_assoc()['total'];
} else {
    $stmt = $conn->prepare(
        "SELECT COUNT(*) total FROM customers WHERE Status!='Active' AND admin_id=?"
    );
    $stmt->bind_param("i", $adminId);
    $stmt->execute();
    $inactiveCustomers = $stmt->get_result()->fetch_assoc()['total'];
}

/*
|--------------------------------------------------------------------------
| PAID CUSTOMERS (CURRENT MONTH)
|--------------------------------------------------------------------------
*/
if ($role === 'SUPER') {
    $paidCustomers = $conn->query("
        SELECT COUNT(DISTINCT customer_id) total
        FROM bills
        WHERE bill_month=$billMonth
          AND bill_year=$billYear
          AND status='PAID'
    ")->fetch_assoc()['total'];
} else {
    $stmt = $conn->prepare("
        SELECT COUNT(DISTINCT b.customer_id) total
        FROM bills b
        JOIN customers c ON b.customer_id=c.id
        WHERE b.bill_month=?
          AND b.bill_year=?
          AND b.status='PAID'
          AND c.admin_id=?
    ");
    $stmt->bind_param("iii", $billMonth, $billYear, $adminId);
    $stmt->execute();
    $paidCustomers = $stmt->get_result()->fetch_assoc()['total'];
}

/*
|--------------------------------------------------------------------------
| UNPAID CUSTOMERS (CURRENT MONTH)
|--------------------------------------------------------------------------
*/
if ($role === 'SUPER') {
    $unpaidCustomers = $conn->query("
        SELECT COUNT(DISTINCT customer_id) total
        FROM bills
        WHERE bill_month=$billMonth
          AND bill_year=$billYear
          AND status!='PAID'
    ")->fetch_assoc()['total'];
} else {
    $stmt = $conn->prepare("
        SELECT COUNT(DISTINCT b.customer_id) total
        FROM bills b
        JOIN customers c ON b.customer_id=c.id
        WHERE b.bill_month=?
          AND b.bill_year=?
          AND b.status!='PAID'
          AND c.admin_id=?
    ");
    $stmt->bind_param("iii", $billMonth, $billYear, $adminId);
    $stmt->execute();
    $unpaidCustomers = $stmt->get_result()->fetch_assoc()['total'];
}

/*
|--------------------------------------------------------------------------
| MONTHLY COLLECTION
|--------------------------------------------------------------------------
*/
if ($role === 'SUPER') {
    $monthlyCollection = $conn->query("
        SELECT IFNULL(SUM(amount),0) total
        FROM payments
        WHERE MONTH(payment_date)=MONTH(CURDATE())
          AND YEAR(payment_date)=YEAR(CURDATE())
    ")->fetch_assoc()['total'];
} else {
    $stmt = $conn->prepare("
        SELECT IFNULL(SUM(p.amount),0) total
        FROM payments p
        JOIN bills b ON p.bill_id=b.id
        JOIN customers c ON b.customer_id=c.id
        WHERE MONTH(p.payment_date)=MONTH(CURDATE())
          AND YEAR(p.payment_date)=YEAR(CURDATE())
          AND c.admin_id=?
    ");
    $stmt->bind_param("i", $adminId);
    $stmt->execute();
    $monthlyCollection = $stmt->get_result()->fetch_assoc()['total'];
}

/*
|--------------------------------------------------------------------------
| TODAY COLLECTION
|--------------------------------------------------------------------------
*/
if ($role === 'SUPER') {
    $dailyCollection = $conn->query("
        SELECT IFNULL(SUM(amount),0) total
        FROM payments
        WHERE DATE(payment_date)=CURDATE()
    ")->fetch_assoc()['total'];
} else {
    $stmt = $conn->prepare("
        SELECT IFNULL(SUM(p.amount),0) total
        FROM payments p
        JOIN bills b ON p.bill_id=b.id
        JOIN customers c ON b.customer_id=c.id
        WHERE DATE(p.payment_date)=CURDATE()
          AND c.admin_id=?
    ");
    $stmt->bind_param("i", $adminId);
    $stmt->execute();
    $dailyCollection = $stmt->get_result()->fetch_assoc()['total'];
}

/*
|--------------------------------------------------------------------------
| TOTAL PENDING AMOUNT
|--------------------------------------------------------------------------
*/
if ($role === 'SUPER') {
    $pendingAmount = $conn->query("
        SELECT IFNULL(SUM(amount - paid_amount),0) total
        FROM bills
        WHERE status!='PAID'
    ")->fetch_assoc()['total'];
} else {
    $stmt = $conn->prepare("
        SELECT IFNULL(SUM(b.amount - b.paid_amount),0) total
        FROM bills b
        JOIN customers c ON b.customer_id=c.id
        WHERE b.status!='PAID'
          AND c.admin_id=?
    ");
    $stmt->bind_param("i", $adminId);
    $stmt->execute();
    $pendingAmount = $stmt->get_result()->fetch_assoc()['total'];
}

/*
|--------------------------------------------------------------------------
| FINAL RESPONSE
|--------------------------------------------------------------------------
*/
echo json_encode([
    "success" => true,
    "data" => [
        "total_customers"     => (int)$totalCustomers,
        "active_customers"    => (int)$activeCustomers,
        "inactive_customers"  => (int)$inactiveCustomers,
        "paid_customers"      => (int)$paidCustomers,
        "unpaid_customers"    => (int)$unpaidCustomers,
        "monthly_collection"  => (float)$monthlyCollection,
        "today_collection"    => (float)$dailyCollection,
        "pending_amount"      => (float)$pendingAmount
    ]
], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
