<?php
header("Content-Type: application/json");

error_reporting(0);
ini_set('display_errors', 0);

include_once "../admin/config/db.php";
include_once "jwt.php";

/*
|--------------------------------------------------------------------------
| JWT AUTH
|--------------------------------------------------------------------------
*/
$headers = getallheaders();
$auth = $headers['Authorization'] ?? '';

if (!$auth || !preg_match('/Bearer\s(\S+)/', $auth, $matches)) {
    http_response_code(401);
    echo json_encode([
        "success" => false,
        "message" => "Unauthorized"
    ]);
    exit;
}

$token = $matches[1];
$user  = verify_jwt($token);

if (!$user || empty($user['role'])) {
    http_response_code(401);
    echo json_encode([
        "success" => false,
        "message" => "Invalid or expired token"
    ]);
    exit;
}

/*
|--------------------------------------------------------------------------
| INPUT
|--------------------------------------------------------------------------
*/
$data = json_decode(file_get_contents("php://input"), true);

$bill_id = intval($data['bill_id'] ?? 0);
$amount  = floatval($data['amount'] ?? 0);

if ($bill_id <= 0 || $amount <= 0) {
    echo json_encode([
        "success" => false,
        "message" => "Invalid bill ID or amount"
    ]);
    exit;
}

/*
|--------------------------------------------------------------------------
| USER CONTEXT
|--------------------------------------------------------------------------
*/
$role    = $user['role'];
$adminId = $user['admin_id'] ?? null;

// Determine who entered payment
if (!empty($user['user_id'])) {
    $enteredBy = $user['user_id'];
} elseif (!empty($user['officer_id'])) {
    $enteredBy = $user['officer_id'];
} elseif (!empty($user['admin_id'])) {
    $enteredBy = $user['admin_id'];
} else {
    echo json_encode([
        "success" => false,
        "message" => "Invalid user context"
    ]);
    exit;
}

/*
|--------------------------------------------------------------------------
| FETCH BILL (SECURITY CHECK)
|--------------------------------------------------------------------------
*/
if ($role === 'SUPER') {

    $sql = "
        SELECT b.id, b.amount, b.paid_amount, c.id AS customer_id
        FROM bills b
        JOIN customers c ON b.customer_id = c.id
        WHERE b.id = ?
        LIMIT 1
    ";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $bill_id);

} else {

    if (!$adminId) {
        echo json_encode([
            "success" => false,
            "message" => "Admin mapping missing"
        ]);
        exit;
    }

    $sql = "
        SELECT b.id, b.amount, b.paid_amount, c.id AS customer_id
        FROM bills b
        JOIN customers c ON b.customer_id = c.id
        WHERE b.id = ?
          AND c.admin_id = ?
        LIMIT 1
    ";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ii", $bill_id, $adminId);
}

$stmt->execute();
$bill = $stmt->get_result()->fetch_assoc();
$stmt->close();

if (!$bill) {
    echo json_encode([
        "success" => false,
        "message" => "Bill not found or access denied"
    ]);
    exit;
}

/*
|--------------------------------------------------------------------------
| PAYMENT CALCULATION
|--------------------------------------------------------------------------
*/
$billAmount = (float)$bill['amount'];
$paidSoFar  = (float)$bill['paid_amount'];
$newPaid    = $paidSoFar + $amount;

$balance = $billAmount - $newPaid;

if ($newPaid >= $billAmount) {
    $status = 'PAID';
} elseif ($newPaid > 0) {
    $status = 'PARTIAL';
} else {
    $status = 'UNPAID';
}

// Carry balance
$carryBalance = 0;
if ($newPaid > $billAmount) {
    $carryBalance = $newPaid - $billAmount;        // advance
} elseif ($newPaid < $billAmount) {
    $carryBalance = -($billAmount - $newPaid);     // due
}

/*
|--------------------------------------------------------------------------
| TRANSACTION
|--------------------------------------------------------------------------
*/
try {
    $conn->begin_transaction();

    // 1️⃣ Insert payment
    $pay = $conn->prepare("
        INSERT INTO payments (bill_id, amount, entered_by)
        VALUES (?, ?, ?)
    ");
    $pay->bind_param("idi", $bill_id, $amount, $enteredBy);
    $pay->execute();
    $pay->close();

    // 2️⃣ INSERT PAYMENT LOG ✅ (NEW)
    $log = $conn->prepare("
        INSERT INTO payment_logs
        (bill_id, customer_id, amount, action, entered_by, role, created_at)
        VALUES (?, ?, ?, 'PAYMENT_COLLECTED', ?, ?, NOW())
    ");
    $log->bind_param(
        "iidis",
        $bill_id,
        $bill['customer_id'],
        $amount,
        $enteredBy,
        $role
    );
    $log->execute();
    $log->close();

    // 3️⃣ Update bill
    $upBill = $conn->prepare("
        UPDATE bills
        SET paid_amount = ?, status = ?
        WHERE id = ?
    ");
    $upBill->bind_param("dsi", $newPaid, $status, $bill_id);
    $upBill->execute();
    $upBill->close();

    // 4️⃣ Update customer carry balance
    $upCust = $conn->prepare("
        UPDATE customers
        SET carry_balance = ?
        WHERE id = ?
    ");
    $upCust->bind_param("di", $carryBalance, $bill['customer_id']);
    $upCust->execute();
    $upCust->close();

    $conn->commit();

    echo json_encode([
        "success" => true,
        "message" => "Payment recorded",
        "data" => [
            "paid_amount"   => $newPaid,
            "status"        => $status,
            "balance"       => max(0, $balance),
            "carry_balance" => $carryBalance
        ]
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

} catch (Exception $e) {

    $conn->rollback();

    echo json_encode([
        "success" => false,
        "message" => "Payment failed"
    ]);
}
