<?php
header("Content-Type: application/json");

include_once "../admin/config/db.php";
include_once __DIR__ . "/jwt.php";

/*
|------------------------------------------------------------
| READ INPUT
|------------------------------------------------------------
*/
$input = json_decode(file_get_contents("php://input"), true);
$refreshToken = trim($input['refresh_token'] ?? '');

if ($refreshToken === '') {
    http_response_code(401);
    echo json_encode([
        "success" => false,
        "message" => "Refresh token required"
    ]);
    exit;
}

/*
|------------------------------------------------------------
| FETCH REFRESH TOKEN FROM DB
|------------------------------------------------------------
*/
$stmt = $conn->prepare("
    SELECT id, user_type, user_id, expires_at
    FROM refresh_tokens
    WHERE token = ?
    LIMIT 1
");
$stmt->bind_param("s", $refreshToken);
$stmt->execute();
$tokenRow = $stmt->get_result()->fetch_assoc();
$stmt->close();

if (!$tokenRow) {

    // 🔴 DEBUG LOG
    error_log("REFRESH FAILED: Invalid token");

    http_response_code(401);
    echo json_encode([
        "success" => false,
        "message" => "Invalid refresh token"
    ]);
    exit;
}

/*
|------------------------------------------------------------
| CHECK EXPIRY
|------------------------------------------------------------
*/
if (strtotime($tokenRow['expires_at']) < time()) {

    // cleanup expired token
    $stmt = $conn->prepare(
        "DELETE FROM refresh_tokens WHERE id = ?"
    );
    $stmt->bind_param("i", $tokenRow['id']);
    $stmt->execute();
    $stmt->close();

    // 🔴 DEBUG LOG
    error_log("REFRESH FAILED: Token expired for user_id=".$tokenRow['user_id']);

    http_response_code(401);
    echo json_encode([
        "success" => false,
        "message" => "Refresh token expired"
    ]);
    exit;
}

/*
|------------------------------------------------------------
| LOAD USER BASED ON ROLE
|------------------------------------------------------------
*/
$userType = $tokenRow['user_type'];
$userId   = (int)$tokenRow['user_id'];

$payload = [];

if ($userType === 'SUPER' || $userType === 'ADMIN') {

    $stmt = $conn->prepare("
        SELECT id, role
        FROM admins
        WHERE id = ?
        LIMIT 1
    ");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $admin = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if (!$admin) {
        error_log("REFRESH FAILED: Admin not found ID=$userId");

        http_response_code(401);
        echo json_encode([
            "success" => false,
            "message" => "Admin not found"
        ]);
        exit;
    }

    $payload = [
        "role"     => strtoupper($admin['role']),
        "admin_id" => (int)$admin['id']
    ];

} elseif ($userType === 'OFFICER') {

    $stmt = $conn->prepare("
        SELECT id, admin_id, status
        FROM officers
        WHERE id = ?
        LIMIT 1
    ");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $officer = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if (!$officer || $officer['status'] !== 'Active') {

        error_log("REFRESH FAILED: Officer inactive ID=$userId");

        http_response_code(401);
        echo json_encode([
            "success" => false,
            "message" => "Officer inactive or not found"
        ]);
        exit;
    }

    $payload = [
        "role"       => "OFFICER",
        "admin_id"   => (int)$officer['admin_id'],
        "officer_id" => (int)$officer['id']
    ];

} else {

    error_log("REFRESH FAILED: Invalid user type");

    http_response_code(401);
    echo json_encode([
        "success" => false,
        "message" => "Invalid user type"
    ]);
    exit;
}

/*
|------------------------------------------------------------
| CREATE NEW ACCESS TOKEN (5 MIN FOR TESTING)
|------------------------------------------------------------
*/
$newAccessToken = create_jwt($payload, 300);

/*
|------------------------------------------------------------
| DEBUG LOG (CONFIRM REFRESH)
|------------------------------------------------------------
*/
error_log(
    "REFRESH SUCCESS: user_type={$userType}, user_id={$userId}"
);

/*
|------------------------------------------------------------
| RESPONSE
|------------------------------------------------------------
*/
echo json_encode([
    "success"       => true,
    "access_token" => $newAccessToken,
    "debug"        => "access_token_refreshed" // 🔧 REMOVE IN PROD
], JSON_UNESCAPED_SLASHES);

exit;
