<?php
header("Content-Type: application/json");

error_reporting(0);
ini_set('display_errors', 0);

include_once "../admin/config/db.php";
include_once __DIR__ . "/jwt.php";

/*
|--------------------------------------------------------------------------
| JWT AUTH
|--------------------------------------------------------------------------
*/
$headers = getallheaders();
$auth = $headers['Authorization'] ?? '';

if (!$auth || !preg_match('/Bearer\s(\S+)/', $auth, $matches)) {
    http_response_code(401);
    echo json_encode(["success" => false, "message" => "Unauthorized"]);
    exit;
}

$user = verify_jwt($matches[1]);

if (!$user || empty($user['role'])) {
    http_response_code(401);
    echo json_encode(["success" => false, "message" => "Invalid or expired token"]);
    exit;
}

$role = strtoupper($user['role']);

/*
|--------------------------------------------------------------------------
| INPUT
|--------------------------------------------------------------------------
*/
$data = json_decode(file_get_contents("php://input"), true);

$bill_id = intval($data['bill_id'] ?? 0);
$amount  = floatval($data['amount'] ?? 0);
$mode    = strtoupper(trim($data['mode'] ?? 'CASH'));

if ($bill_id <= 0 || $amount <= 0) {
    echo json_encode([
        "success" => false,
        "message" => "Invalid bill ID or amount"
    ]);
    exit;
}

/*
|--------------------------------------------------------------------------
| USER CONTEXT
|--------------------------------------------------------------------------
*/
$adminId   = null;
$officerId = null;
$enteredBy = null;

/* ===== ADMIN LOGIN ===== */
if (!empty($user['admin_id']) && empty($user['officer_id'])) {

    $adminId   = (int)$user['admin_id'];
    $enteredBy = $adminId;
    $officerId = null;

}
/* ===== OFFICER LOGIN ===== */
elseif (!empty($user['officer_id'])) {

    $officerId = (int)$user['officer_id'];

    // Find parent admin
    $stmt = $conn->prepare("
        SELECT admin_id 
        FROM officers 
        WHERE id = ? 
        LIMIT 1
    ");
    $stmt->bind_param("i", $officerId);
    $stmt->execute();
    $stmt->bind_result($adminId);
    $stmt->fetch();
    $stmt->close();

    if (!$adminId) {
        echo json_encode([
            "success" => false,
            "message" => "Officer admin mapping missing"
        ]);
        exit;
    }

    $enteredBy = $adminId;

} else {
    echo json_encode([
        "success" => false,
        "message" => "Invalid user context"
    ]);
    exit;
}

/*
|--------------------------------------------------------------------------
| FETCH BILL (SECURITY)
|--------------------------------------------------------------------------
*/
if ($role === 'SUPER') {

    $stmt = $conn->prepare("
        SELECT b.id, b.amount, b.paid_amount, c.id AS customer_id
        FROM bills b
        JOIN customers c ON b.customer_id = c.id
        WHERE b.id = ?
        LIMIT 1
    ");
    $stmt->bind_param("i", $bill_id);

} else {

    $stmt = $conn->prepare("
        SELECT b.id, b.amount, b.paid_amount, c.id AS customer_id
        FROM bills b
        JOIN customers c ON b.customer_id = c.id
        WHERE b.id = ?
          AND c.admin_id = ?
        LIMIT 1
    ");
    $stmt->bind_param("ii", $bill_id, $adminId);
}

$stmt->execute();
$bill = $stmt->get_result()->fetch_assoc();
$stmt->close();

if (!$bill) {
    echo json_encode([
        "success" => false,
        "message" => "Bill not found or access denied"
    ]);
    exit;
}

/*
|--------------------------------------------------------------------------
| PAYMENT CALCULATION
|--------------------------------------------------------------------------
*/
$billAmount = (float)$bill['amount'];
$paidSoFar  = (float)$bill['paid_amount'];

$newPaid = $paidSoFar + $amount;
$balance = $billAmount - $newPaid;

if ($newPaid >= $billAmount) {
    $status = 'PAID';
} elseif ($newPaid > 0) {
    $status = 'PARTIAL';
} else {
    $status = 'UNPAID';
}

$carryBalance = $newPaid - $billAmount;

/*
|--------------------------------------------------------------------------
| TRANSACTION
|--------------------------------------------------------------------------
*/
try {
    $conn->begin_transaction();

    // 1️⃣ Insert payment
    $stmt = $conn->prepare("
        INSERT INTO payments
        (bill_id, amount, payment_date, mode, entered_by, admin_id, officer_id)
        VALUES (?, ?, NOW(), ?, ?, ?, ?)
    ");
    $stmt->bind_param(
        "idssii",
        $bill_id,
        $amount,
        $mode,
        $enteredBy,
        $adminId,
        $officerId
    );
    $stmt->execute();
    $stmt->close();

    // 2️⃣ Insert payment log
    $roleLabel = !empty($officerId) ? 'officer' : 'admin';

    $stmt = $conn->prepare("
        INSERT INTO payment_logs
        (bill_id, customer_id, amount, action, entered_by, role, mode, admin_id, created_at)
        VALUES (?, ?, ?, 'PAYMENT', ?, ?, ?, ?, NOW())
    ");
    $stmt->bind_param(
        "iidissi",
        $bill_id,
        $bill['customer_id'],
        $amount,
        $enteredBy,
        $roleLabel,
        $mode,
        $adminId
    );
    $stmt->execute();
    $stmt->close();

    // 3️⃣ Update bill
    $stmt = $conn->prepare("
        UPDATE bills
        SET paid_amount = ?, status = ?
        WHERE id = ?
    ");
    $stmt->bind_param("dsi", $newPaid, $status, $bill_id);
    $stmt->execute();
    $stmt->close();

    // 4️⃣ Update customer carry balance
    $stmt = $conn->prepare("
        UPDATE customers
        SET carry_balance = ?
        WHERE id = ?
    ");
    $stmt->bind_param("di", $carryBalance, $bill['customer_id']);
    $stmt->execute();
    $stmt->close();

    $conn->commit();

    echo json_encode([
        "success" => true,
        "message" => "Payment recorded successfully",
        "data" => [
            "status"        => $status,
            "paid_amount"   => $newPaid,
            "balance"       => max(0, $balance),
            "carry_balance" => $carryBalance,
            "mode"          => $mode
        ]
    ]);

} catch (Exception $e) {

    $conn->rollback();

    echo json_encode([
        "success" => false,
        "message" => "Payment failed"
    ]);
}
